﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecatalyst/CodeCatalystRequest.h>
#include <aws/codecatalyst/CodeCatalyst_EXPORTS.h>
#include <aws/codecatalyst/model/IdeConfiguration.h>
#include <aws/codecatalyst/model/InstanceType.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CodeCatalyst {
namespace Model {

/**
 */
class UpdateDevEnvironmentRequest : public CodeCatalystRequest {
 public:
  AWS_CODECATALYST_API UpdateDevEnvironmentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateDevEnvironment"; }

  AWS_CODECATALYST_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the space.</p>
   */
  inline const Aws::String& GetSpaceName() const { return m_spaceName; }
  inline bool SpaceNameHasBeenSet() const { return m_spaceNameHasBeenSet; }
  template <typename SpaceNameT = Aws::String>
  void SetSpaceName(SpaceNameT&& value) {
    m_spaceNameHasBeenSet = true;
    m_spaceName = std::forward<SpaceNameT>(value);
  }
  template <typename SpaceNameT = Aws::String>
  UpdateDevEnvironmentRequest& WithSpaceName(SpaceNameT&& value) {
    SetSpaceName(std::forward<SpaceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the project in the space.</p>
   */
  inline const Aws::String& GetProjectName() const { return m_projectName; }
  inline bool ProjectNameHasBeenSet() const { return m_projectNameHasBeenSet; }
  template <typename ProjectNameT = Aws::String>
  void SetProjectName(ProjectNameT&& value) {
    m_projectNameHasBeenSet = true;
    m_projectName = std::forward<ProjectNameT>(value);
  }
  template <typename ProjectNameT = Aws::String>
  UpdateDevEnvironmentRequest& WithProjectName(ProjectNameT&& value) {
    SetProjectName(std::forward<ProjectNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The system-generated unique ID of the Dev Environment. </p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  UpdateDevEnvironmentRequest& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user-specified alias for the Dev Environment. Changing this value will
   * not cause a restart.</p>
   */
  inline const Aws::String& GetAlias() const { return m_alias; }
  inline bool AliasHasBeenSet() const { return m_aliasHasBeenSet; }
  template <typename AliasT = Aws::String>
  void SetAlias(AliasT&& value) {
    m_aliasHasBeenSet = true;
    m_alias = std::forward<AliasT>(value);
  }
  template <typename AliasT = Aws::String>
  UpdateDevEnvironmentRequest& WithAlias(AliasT&& value) {
    SetAlias(std::forward<AliasT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the integrated development environment (IDE) configured for
   * a Dev Environment.</p>
   */
  inline const Aws::Vector<IdeConfiguration>& GetIdes() const { return m_ides; }
  inline bool IdesHasBeenSet() const { return m_idesHasBeenSet; }
  template <typename IdesT = Aws::Vector<IdeConfiguration>>
  void SetIdes(IdesT&& value) {
    m_idesHasBeenSet = true;
    m_ides = std::forward<IdesT>(value);
  }
  template <typename IdesT = Aws::Vector<IdeConfiguration>>
  UpdateDevEnvironmentRequest& WithIdes(IdesT&& value) {
    SetIdes(std::forward<IdesT>(value));
    return *this;
  }
  template <typename IdesT = IdeConfiguration>
  UpdateDevEnvironmentRequest& AddIdes(IdesT&& value) {
    m_idesHasBeenSet = true;
    m_ides.emplace_back(std::forward<IdesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon EC2 instace type to use for the Dev Environment. </p>
   * <p>Changing this value will cause a restart of the Dev Environment if it is
   * running.</p>
   */
  inline InstanceType GetInstanceType() const { return m_instanceType; }
  inline bool InstanceTypeHasBeenSet() const { return m_instanceTypeHasBeenSet; }
  inline void SetInstanceType(InstanceType value) {
    m_instanceTypeHasBeenSet = true;
    m_instanceType = value;
  }
  inline UpdateDevEnvironmentRequest& WithInstanceType(InstanceType value) {
    SetInstanceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The amount of time the Dev Environment will run without any activity detected
   * before stopping, in minutes. Only whole integers are allowed. Dev Environments
   * consume compute minutes when running.</p>  <p>Changing this value will
   * cause a restart of the Dev Environment if it is running.</p>
   */
  inline int GetInactivityTimeoutMinutes() const { return m_inactivityTimeoutMinutes; }
  inline bool InactivityTimeoutMinutesHasBeenSet() const { return m_inactivityTimeoutMinutesHasBeenSet; }
  inline void SetInactivityTimeoutMinutes(int value) {
    m_inactivityTimeoutMinutesHasBeenSet = true;
    m_inactivityTimeoutMinutes = value;
  }
  inline UpdateDevEnvironmentRequest& WithInactivityTimeoutMinutes(int value) {
    SetInactivityTimeoutMinutes(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A user-specified idempotency token. Idempotency ensures that an API request
   * completes only once. With an idempotent request, if the original request
   * completes successfully, the subsequent retries return the result from the
   * original successful request and have no additional effect.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  UpdateDevEnvironmentRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_spaceName;

  Aws::String m_projectName;

  Aws::String m_id;

  Aws::String m_alias;

  Aws::Vector<IdeConfiguration> m_ides;

  InstanceType m_instanceType{InstanceType::NOT_SET};

  int m_inactivityTimeoutMinutes{0};

  Aws::String m_clientToken;
  bool m_spaceNameHasBeenSet = false;
  bool m_projectNameHasBeenSet = false;
  bool m_idHasBeenSet = false;
  bool m_aliasHasBeenSet = false;
  bool m_idesHasBeenSet = false;
  bool m_instanceTypeHasBeenSet = false;
  bool m_inactivityTimeoutMinutesHasBeenSet = false;
  bool m_clientTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeCatalyst
}  // namespace Aws
